<?php

namespace PureCharity\Fundraisers;

use \stdClass;

class PublicController extends FundraisersController {

    public $moduleConfig;

    const DATE_FORMAT = "M j, Y";
    const MONTH_FORMAT = "M Y";

	public static $fundraiser;
	public static $fundraisers;
	public static $options;
	public static $includes;

    private $version;
    private $pluginName;
    
    public static $classPath = '\PureCharity\Fundraisers\PublicController';

    function __construct($moduleConfig) {
        $this->moduleConfig = $moduleConfig;
    
        $this->pluginName = $this->moduleConfig['plugin']['name'];
        $this->version = $this->moduleConfig['plugin']['version'];

		self::$includes = [
			'fundingBar' => $this->moduleConfig['base']['path'] .'/includes/fundraisers/single-view-funding-bar.php',
			'fundingDiv' => $this->moduleConfig['base']['path'] .'/includes/fundraisers/single-view-funding-div.php',
			'tabs' => $this->moduleConfig['base']['path'] .'/includes/fundraisers/single-view-tabs.php',
		];
    }

    /**
	 * Register the stylesheets for the Dashboard.
	 *
	 * 
	 */
	public function enqueueStyles() 
	{

		if (self::moduleResourceExists('fundraisers')) {
		
			wp_enqueue_style(
				'purecharity-plugin-fundraisers-public', 
				$this->moduleConfig['resources']['styles'] .'public/fundraisers.css', 
				[], 
				$this->version, 
				'all'
			);

			self::handleModuleResource('fundraisers', false);
		}
	}

    /**
	 * Register the JavaScript for the dashboard.
	 *
	 * 
	 */
	public function enqueueScripts($hook) 
	{
	}

	public static function listNotFound($default = true){
		$html = '<p class="fr-not-found" style="'. ( $default ? '' : 'display:none' ) .'">No Fundraisers Found.</p>';
		return $html;
	}

	public static function notFound(){

		global $PcPlugin;
		$baseModule = $PcPlugin->getModule('base', 'BaseController');
		$basePublicController = $baseModule->getController('PublicController');

		return "<p>Fundraiser Not Found.</p>" . $basePublicController::poweredBy();
	}

	public static function listing() {

		$options = get_option( 'purecharity_plugin_fundraisers_settings' );

		global $PcPlugin;
		$baseModule = $PcPlugin->getModule('base', 'BaseController');
		$basePublicController = $baseModule->getController('PublicController');

		$fundraisersModule = $PcPlugin->getModule('fundraisers', 'FundraisersController');
		$paginatorController = $fundraisersModule->getController('PaginatorController');
	
		$html = self::printCustomStyles() ;
		$html .= '
		  <div class="fr-list-container">
			'.self::liveSearch().'
			<table class="fundraiser-table option1">
			  <tr>
				  <th colspan="2">Fundraiser Name</th>
				</tr>
		';
		$i = 0;
	
		$used = array();
		foreach(self::$fundraisers->external_fundraisers as $fundraiser){
		  if(!in_array($fundraiser->id, $used)){
			array_push($used, $fundraiser->id);
			$title = $fundraiser->name;
			if(isset(self::$options['title']) && self::$options['title'] == 'owner_name'){
			  $title = $fundraiser->owner->name;
			}
			if(isset(self::$options['title']) && self::$options['title'] == 'title_and_owner_name'){
			  $title = $fundraiser->name.' by '.$fundraiser->owner->name;
			}
	
			$class = $i&1 ? 'odd' : 'even';
			$i += 1;
			$html .= '
			  <tr class="row '.$class.' fundraiser_'.$fundraiser->id.'">
				  <td>'.$title.'</td>
				  <td>
					<a class="fr-themed-link pc-fundraiser-more-info" href="?fundraiser='. $fundraiser->slug .'&id='. $fundraiser->id .'&campaign='. self::$options['campaign'] .'">More Info</a>
					<a class="donate" href="'. $basePublicController::pcUrl() .'/fundraisers/'. $fundraiser->id .'/fund">Donate Now</a>
				</td>
			   </tr>
			';
		  }
		}
	
		  $html .= '
		  </table>
			'. self::listNotFound(false) .'
		  </div>
		';

		$html .= $paginatorController::pageLinks(self::$fundraisers->meta);
		$html .= $basePublicController::poweredBy();
		
		return $html;
	}

	public static function printCustomStyles() {
		$base_settings = get_option( 'purecharity_plugin_settings' );
		$pf_settings = get_option( 'purecharity_plugin_fundraisers_settings' );

		// Default theme color
		if($pf_settings['main_color'] == NULL || $pf_settings['main_color'] == ''){
		  if($base_settings['main_color'] == NULL || $base_settings['main_color'] == ''){
			$color = '#CA663A';
		  }else{
			$color = $base_settings['main_color'];
		  }
		}else{
		  $color = $pf_settings['main_color'];
		}
	
		$html = '<style id="pc-fr-page-container-flag">';
		$html .= '
		  .fundraiser-table a.donate { background: '.$color.' !important; }
		  .fr-grid-progress { background: '.$color.' !important; }
		  .fr-grid-list-item ul.fr-list-actions li a:hover { background: '.$color.' !important; }
		  a.fr-pure-button { background: '.$color.' !important; }
		  .fr-single-progress { background: '.$color.' !important; }
		  #fr-tabs ul.fr-tabs-list li.active a, #fr-tabs ul.fr-tabs-list li a:hover {border-color: '.$color.' !important;}
		  .fr-themed-link { color: '.$color.' !important; }
		  .fr-filtering button,
		  .fr-filtering a { background: '.$color.' }
		';
		$html .= '</style>';
	
		return $html;
	}

	public static function liveSearch() {

		$options = get_option( 'purecharity_plugin_fundraisers_settings' );
	
		if(isset($options["live_filter"]) && (empty(self::$options['hide_search']) || self::$options['hide_search'] != 'true')){
		  $html = '
			<div class="fr-filtering pure_col pure_span_24">
			  <form method="get">
				<fieldset class="livefilter fr-livefilter">
				  <legend>
					<label for="livefilter-input">
					  <strong>Search Fundraisers:</strong>
					</label>
				  </legend>
				  <input id="livefilter-input" class="fr-livefilter-input" value="'.@$_GET['query'].'" name="query" type="text">
				  <button class="fr-filtering-button" type="submit">Filter</button>
				  '. (@$_GET['query'] != '' ? '<a href="#" class="pc-fr-clear-filter">Clear filter</a>' : '') .'
				</fieldset>
			  </form>
			</div>
		  ';
		}else{
		  $html = '';
		}
		return $html;
	}	

	public static function show() {

		global $PcPlugin;
		$baseModule = $PcPlugin->getModule('base', 'BaseController');
		$basePublicController = $baseModule->getController('PublicController');

		$title = self::$fundraiser->name;
		if(isset(self::$options['title']) && self::$options['title'] == 'owner_name'){
		  $title = self::$fundraiser->owner->name;
		}
		if(isset(self::$options['title']) && self::$options['title'] == 'title_and_owner_name'){
		  $title = self::$fundraiser->name.' by '.self::$fundraiser->owner->name;
		}
	
		self::$options['external_url'] = $basePublicController::pcUrl() .'/fundraisers/'. self::$fundraiser->id .'/fund';
		$share_url = "http://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";
		$summary = self::$fundraiser->name .' is raising money for their adoption from '. self::$fundraiser->country;
	
		$options = get_option( 'purecharity_plugin_fundraisers_settings' );
	
		$html = self::printCustomStyles();
	
		self::$fundraiser->description = !empty(self::$fundraiser->description) ? self::$fundraiser->description : '';

		if(isset(self::$options["layout"]) && ((int)self::$options['layout'] == 4)){
		  # ADOPTIONS ONLY
		  $html .= '
			<div class="wrapper">
			  <div class="pure_row families show">
				<div class="pure_col pure_span_16">
				  <div class="family-details">
					<img  alt="'. self::$fundraiser->name .' is raising money for their adoption from '. self::$fundraiser->country .'."
						  class="img-responsive"
						  src="'. self::$fundraiser->images->large .'">
					<h1 class="title">'. self::$fundraiser->name .'</h1>
					'. self::grid4Pieces('adopting_from_show') .'
					<div class="description"><p>'. self::$fundraiser->description .'</p></div>
					<div class="pure_col">
					  '. self::grid4Pieces('adoption_status') .'
					  '. self::grid4Pieces('adoption_agency') .'
					</div>
					'. self::grid4Pieces('about') .'
					'. self::grid4Pieces('updates') .'
				  </div>
				</div>
	
				<aside class="pure_col pure_span_8">
				  '. self::grid4Pieces('funding') .'
				  <div class="share-buttons">
					<div class="pure_row no-padding">
					  <div class="pure_col pure_span_12 share-button">
						<a class="facebook" href="https://www.facebook.com/sharer/sharer.php?u='. $share_url .'">Facebook</a>
					  </div>
					  <div class="pure_col pure_span_12 share-button">
						<a class="twitter" href="https://twitter.com/home?status='. $summary .'%21%20'. $share_url .'">Twitter</a>
					  </div>
					</div>
					<div class="pure_row no-padding">
					  <div class="pure_col pure_span_12 share-button">
						<a class="google-plus" href="https://plus.google.com/share?url='. $share_url .'">Google +</a>
					  </div>
					  <div class="pure_col pure_span_12 share-button">
						<a class="linkedin" href="https://www.linkedin.com/shareArticle?mini=true&url='. $share_url .'&summary='. $summary .'&source=">LinkedIn</a>
					  </div>
					</div>
					<div class="pure_row no-padding">
					  <div class="pure_col pure_span_12 share-button">
					  <a class="pinterest" href="https://pinterest.com/pin/create/button/?url='. $share_url .'&media='. self::$fundraiser->images->large .'&description='. $summary .'}">Pinterest</a>
					  </div>
					  <div class="pure_col pure_span_12 share-button">
					  <a class="email" href="mailto:?&subject=Check the '. self::$fundraiser->name .' fundraising campaign&body='. $summary .'%0A%0AThanks!">Email</a>
					  </div>
					</div>
					<input type="text" name="url" id="url" value="'. $share_url .'" class="form-control" onclick="this.setSelectionRange(0, this.value.length)">
				  </div>
				  '. self::grid4Pieces('backers') .'
				</aside>
	
			  </div>
			</div>
		  ';
		}else{

			$options = get_option( 'purecharity_plugin_fundraisers_settings' );
			$htmlExtension = '';
			
			if (!isset($options['organized_by'])) {
				$htmlExtension .= '<p class="fr-organizer">
						Organized by <a class="fr-themed-link" href="'. $basePublicController::pcUrl() .'/'. self::$fundraiser->field_partner_slug .'">'.self::$fundraiser->owner->name.'</a>
					  </p>
						<style>.fr-location { padding-bottom: 0; }</style>
						';
			}

			$withMapIcon = (
				isset($options['map_icon']) ?
				'hide-icon' :
				''
			);

		  $html .= '
			<div class="pure_row">
			  <div class="fr-top-row pure_col pure_span_24">
				<div class="fr-name pure_col pure_span_18">
				  <h3>'.$title.'</h3>
				</div>
				<div class="fr-donate mobile-hidden fr-donate-top pure_col pure_span_6">
				  <a class="fr-pure-button" href="'. $basePublicController::pcUrl() .'/fundraisers/'. self::$fundraiser->id .'/fund">Donate</a>
				</div>
			  </div>
			  <div class="fr-container pure_col pure_span_24 fundraiser_'.self::$fundraiser->id.'">
				<div class="fr-header pure_col pure_span_24">
				  <img src="'.self::$fundraiser->images->large.'">
				</div>
				<div class="fr-middle-row pure_col pure_span_24">
				  <div class="fr-avatar-container pure_col pure_span_5">
					<div class="fr-avatar" href="#" style="background-image: url('.self::$fundraiser->images->small.')"></div>
				  </div>
				  <div class="fr-info pure_col pure_span_13">
					<p class="fr-location '. $withMapIcon .'">'.self::$fundraiser->country.'</p>
					'. $htmlExtension .'
				  </div>
				  <div class="fr-donate pure_col pure_span_6">
					<a class="fr-pure-button" href="'. $basePublicController::pcUrl() .'/fundraisers/'. self::$fundraiser->id.'/fund">Donate</a>
					'. (isset($options['fundraise_cause']) ?  '' : '<a class="fr-p2p" href="'. $basePublicController::pcUrl() .'/'. self::$fundraiser->slug.'/copies/new">Start a Fundraiser for this Cause</a>') .'
				  </div>
				</div>
				'. self::singleViewFundingBar() .'
				'. self::singleViewFundingDiv() .'
				'. self::singleViewTabs() .'
			  </div>
			</div>
		  ';
		}

		$html .= $basePublicController::poweredBy(isset($options['powered_by_pure_charity']));
		return $html;
	}

	public static function singleViewFundingBar(){
		include(self::$includes['fundingBar']);
		return $html;
	}

	public static function singleViewFundingDiv(){
		include(self::$includes['fundingDiv']);
		return $html;
	}

	public static function singleViewTabs(){
		include(self::$includes['tabs']);
		return $html;
	}

	public static function percent($num_amount, $num_total) {
		if($num_total == 0){ return 100; }
		return number_format((($num_amount / $num_total) * 100), 0);
	}

	public static function printUpdates() {
		if(empty(self::$fundraiser->updates)){
		  $html = '<p>There are no updates at this time.</p>';
		}else{
			global $PcPlugin;
			$baseModule = $PcPlugin->getModule('base', 'BaseController');
			$basePublicController = $baseModule->getController('PublicController');
			
		  $html = '<ul class="fr-updates">';
		  foreach(self::$fundraiser->updates as $update){
			$html .= '
			  <li>
				  <h4><a class="fr-themed-link" href="'.$update->url.'">'.$update->title.'</a></h4>
				  <p class="date">Posted a week ago</p>
				  <p>'.$update->body.'</p>
				  <span class="fr-author">
					<p>Posted by:<br/><a class="fr-themed-link" href="'. $basePublicController::pcUrl() .'/'.$update->author->slug.'">'.$update->author->name.'</a></p>
				  </span>
				  <span class="fr-read-more">
					<a class="fr-read-more" href="'.$update->url.'">Read More</a><!-- links to update on pure charity -->
				  </span>
				</li>
			';
		  }
		  $html .= '</ul>';
		}
		return $html;
	}

	public static function printBackers() {
		if(empty(self::$fundraiser->public_backers)){
		  $html = '<p>There are no backers at this time.</p>';
		}else{
			global $PcPlugin;
			$baseModule = $PcPlugin->getModule('base', 'BaseController');
			$basePublicController = $baseModule->getController('PublicController');
		  $html = '<ul class="fr-backers pure_col pure_span_24">';
		  foreach(self::$fundraiser->public_backers as $backer){
			$html .= '
			  <li class="pure_col pure_span_6">
				<span class="fr-avatar fr-backer-avatar" href="#" style="background-image: url('.$backer->avatar.')"></span>
				<span class="fr-backer-name"><a class="fr-themed-link" href="'. $basePublicController::pcUrl() .'/'.$backer->slug.'">'.$backer->name.'</a></span>
			  </li>
			';
		  }
		  $html .= '</ul>';
		}
		return $html;
	}

	/**
	 * Confitional layout pieces for layout 4 (grid, single view)
	 *
	 * 
	 */
	public static function grid4Pieces($piece = null){
		
		if($piece == null) { return ''; }
		
		self::$fundraiser->backers = !empty(self::$fundraiser->backers) ? self::$fundraiser->backers : [];

		$html = '';

		switch($piece){

		case 'funding_list':
			$html = '';
			if(is_int(self::$fundraiser->funding_goal)){
			$html = '<span class="raised">$'. number_format(self::$fundraiser->funding_goal-self::$fundraiser->funding_needed, 2).' Raised</span>';
			}
			break;

		case 'funding':
			$html = '<div class="raised">';
			if(is_int(self::$fundraiser->funding_goal)){
			$html .= '
				<h3>Raised</h3>
				<span class="total-raised"> $'. number_format((self::$fundraiser->funding_goal-self::$fundraiser->funding_needed), 2) .'</span>
				<span class="goal">of $'. number_format(self::$fundraiser->funding_goal, 2) .' Goal</span>
			';
			}
			$html .= '
			<a class="pcbtn pcbtn-primary pcbtn-lg pcbtn-block" href="'. self::$options['external_url'] .'">Give to this Adoption</a>
			</div>';

			break;

		case 'adopting_from':
			if(!empty(self::$fundraiser->country)){
			$html = '<span class="location">is adopting from '. self::$fundraiser->country .'</span>';
			}
			break;

		case 'adopting_from_show':
			if(!empty(self::$fundraiser->country)){
			$html = '<h2 class="subtitle">is adopting a child from '. self::$fundraiser->country .'</h2>';
			}
			break;


		case 'adoption_status':
			if(!empty(self::$fundraiser->adoption_status)){
			$html = '
				<div class="pure_span_12">
				<h4>Adoption Status</h4>
				<h3>'. self::$fundraiser->adoption_status .'</h3>
				</div><hr>';
			}
			break;

		case 'adoption-agency':
			if(!empty(self::$fundraiser->beneficiary_field_partner_name)){
			$website = self::$fundraiser->beneficiary_field_partner_website;
			$html = '
			<div class="pure_span_12 agency">
				<h4>Adoption Agency</h4>
				<h3>'. self::$fundraiser->beneficiary_field_partner_name .'</h3>';
				if(!empty($website)){
				$html .= '<span class="website"><a target="_blank" href="'.$website.'">'.$website.'</a></span>';
				}
			$html .= '</div><hr>';
			}
			break;

		case 'about':
			if(!empty(self::$fundraiser->about)){
			$html = '
				<hr>
				<div class="row">
				<div class="pure_span_24">
					<h2>About</h2>
					<p>'. self::$fundraiser->about .'</p>
				</div>
				</div>';
			}
			break;

		case 'updates':
			$html = '';
			foreach(self::$fundraiser->updates as $update){
			$html .= '
				<div class="row">
				<div class="pure_span_24">
					<hr>
					<h3>
					<a href="'. $update->title .'">'. $update->title .'</a>
					<small>by '. $update->author->name .'</small>
					</h3>
					<p class="overflow-hidden">'. $update->body .'</p>
				</div>
				</div>
			';
			}
			break;

		case 'backers':
			$count = count(self::$fundraiser->backers);
			if($count > 0){
			$html = '<h2>'. $count . ' ' . self::pluralize(count(self::$fundraiser->backers), 'Donation') .' <small>from:</small></h2>';
			$html .= '<ul class="list-unstyled">';
			foreach(self::$fundraiser->backers as $backer){
				$html .= '
				<li class="donor">
					<h5>'. $backer->name .'</h5>
				</li>
				';
			}
			$html .= '</ul>';
			}
			break;
		}

		return $html;
	}

	/**
	 * List of Fundraisers.
	 *
	 * @since    1.0.0
	*/
	public static function listingGrid($opts){

		$layout = empty($opts['layout']) ? 1 : $opts['layout'];

		self::$options = get_option( 'purecharity_plugin_fundraisers_settings' );

		self::$options['campaign'] = (
			!empty($opts['campaign']) ?
			$opts['campaign'] :
			null
		);

		switch ((int) $layout) {
		case 1:
			return self::gridOption1();
			break;
		case 2:
			return self::gridOption2();
			break;
		case 3:
			return self::gridOption3();
			break;
		case 4:
			return self::gridOption4();
			break;
		default:
			return self::gridOption1();
			break;
		}
	}

	/**
	 * Grid listing layout option 1.
	 *
	 * 
	*/
	public static function gridOption1(){
		$html = self::printCustomStyles() ;
		$html .= '<div class="fr-list-container pure_centered pure_row is-grid">'.self::liveSearch();
		$html .= '<div>';

		global $PcPlugin;
		$baseModule = $PcPlugin->getModule('base', 'BaseController');
		$basePublicController = $baseModule->getController('PublicController');

		$fundraisersModule = $PcPlugin->getModule('fundraisers', 'FundraisersController');
		$paginatorController = $fundraisersModule->getController('PaginatorController');

		$used = array();
		$counter = 1;
		foreach(self::$fundraisers->external_fundraisers as $fundraiser){
		if(!in_array($fundraiser->id, $used)){
			array_push($used, $fundraiser->id);

			$fundraiser->funding_goal = (
				$fundraiser->funding_goal !== 'anonymous' ?
				$fundraiser->funding_goal :
				$fundraiser->funding_total
			);
			
			$fundraiser->funding_needed = (
				$fundraiser->funding_needed !== 'anonymous' ?
				$fundraiser->funding_needed :
				$fundraiser->funding_total
			);

			$title = $fundraiser->name;
			if(isset(self::$options['title']) && self::$options['title'] == 'owner_name'){
			$title = $fundraiser->owner->name;
			}
			if(isset(self::$options['title']) && self::$options['title'] == 'title_and_owner_name'){
			$title = $fundraiser->name.'<br /> by '.$fundraiser->owner->name;
			}

			if ($fundraiser->images->medium == NULL) {
			$image = $fundraiser->images->large;
			}else{
			$image = $fundraiser->images->medium;
			}

			$funded = self::percent(($fundraiser->funding_goal-$fundraiser->funding_needed) ,$fundraiser->funding_goal);
			$html .= '
			<div class="fr-grid-list-item pure_span_6 pure_col fundraiser_'.$fundraiser->id.'">
				<div class="fr-grid-list-content">
			';

			$html .= '
			<div class="fr-listing-avatar-container pure_span24">
				<div class="fr-listing-avatar" href="#" style="background-image: url('.$image.')">
				<a href="?fundraiser='.$fundraiser->slug.'&campaign='. self::$options['campaign'] .'" class="overlay-link"></a>
				</div>
			</div>
			';

			$html .='
			<div class="fr-grid-item-content pure_col pure_span_24">
				<div class="fr-grid-title-container">
				<p class="fr-grid-title">'.$title.'</p>
				<p class="fr-grid-desc">'.strip_tags(self::truncate($fundraiser->about, 100)).'</p>
				</div>
				'.self::gridFundingStats($fundraiser).'
			</div>
			<div class="fr-actions pure_col pure_span_24">
				<a class="fr-themed-link pc-fundraiser-more-info" href="?fundraiser='.$fundraiser->slug.'&campaign='. self::$options['campaign'] .'">More</a>
				<a class="fr-themed-link" href="'.$basePublicController::pcUrl().'/fundraisers/'.$fundraiser->id.'/fund">Donate</a>
			</div>
			';

			$html .= '
				</div>
			</div>
			';
			if($counter %4 == 0){
			$html .= '<hr class="hidden"></hr>';
			}
			$counter ++;
		}
		}

		$html .= self::listNotFound(!!(count($used)==0));
		$html .= '</div>';
		$html .= '</div>';
		$html .= $paginatorController::pageLinks(self::$fundraisers->meta);
		$html .= $basePublicController::poweredBy();

		return $html;
	}

	/**
	 * Grid listing layout option 2.
	 *
	 * 
	 */
	public static function gridOption2() {
		$html = self::printCustomStyles() ;
		$html .= '<div class="fr-list-container pure_centered pure_row is-grid">'.self::liveSearch();
		$html .= '<div>';

		global $PcPlugin;
		$baseModule = $PcPlugin->getModule('base', 'BaseController');
		$basePublicController = $baseModule->getController('PublicController');

		$fundraisersModule = $PcPlugin->getModule('fundraisers', 'FundraisersController');
		$paginatorController = $fundraisersModule->getController('PaginatorController');

		$used = array();
		$counter = 1;
		foreach(self::$fundraisers->external_fundraisers as $fundraiser){
		if(!in_array($fundraiser->id, $used)){
			array_push($used, $fundraiser->id);

			$fundraiser->funding_goal = (
				$fundraiser->funding_goal !== 'anonymous' ?
				$fundraiser->funding_goal :
				$fundraiser->funding_total
			);
			
			$fundraiser->funding_needed = (
				$fundraiser->funding_needed !== 'anonymous' ?
				$fundraiser->funding_needed :
				$fundraiser->funding_total
			);

			$title = $fundraiser->name;
			if(isset(self::$options['title']) && self::$options['title'] == 'owner_name'){
			$title = $fundraiser->owner->name;
			}
			if(isset(self::$options['title']) && self::$options['title'] == 'title_and_owner_name'){
			$title = $fundraiser->name.'<br /> by '.$fundraiser->owner->name;
			}

			if ($fundraiser->images->medium == NULL) {
			$image = $fundraiser->images->large;
			}else{
			$image = $fundraiser->images->medium;
			}

			$funded = self::percent(($fundraiser->funding_goal-$fundraiser->funding_needed) ,$fundraiser->funding_goal);
			$html .= '
			<div class="fr-grid-list-item pure_span_8 pure_col fundraiser_'.$fundraiser->id.'">
				<div class="fr-grid-list-content">
				<div class="fr-listing-avatar-container extended pure_span24">
					<div class="fr-listing-avatar" style="background-image: url('.$image.')">
						<a href="?fundraiser='.$fundraiser->slug.'&campaign='. self::$options['campaign'] .'" class="overlay-link pc-fundraiser-more-info"></a>
					</div>
				</div>
				<div class="fr-grid-item-content pure_col pure_span_24">
					<div class="fr-grid-title-container">
					<p class="fr-grid-title extended">'.$title.'</p>
					<p class="fr-grid-desc extended">'.strip_tags(self::truncate($fundraiser->about, 150)).'</p>
					</div>
					'.self::gridFundingStats($fundraiser, 2).'
				</div>
				<div class="fr-actions extended pure_col pure_span_24">
					<a class="fr-themed-link pc-fundraiser-more-info" href="?fundraiser='.$fundraiser->slug.'&campaign='. self::$options['campaign'] .'">More</a>
					<a class="fr-themed-link" target="_blank" href="'.$basePublicController::pcUrl().'/fundraisers/'.$fundraiser->id.'/fund">Donate</a>
				</div>
				</div>
			</div>
			';
			if($counter %3 == 0){
			$html .= '<div class="clearfix"></div>';
			}
			$counter ++;
		}
		}

		$html .= self::listNotFound(!!(count($used)==0));
		$html .= '</div>';
		$html .= '</div>';
		$html .= $paginatorController::pageLinks(self::$fundraisers->meta);
		$html .= $basePublicController::poweredBy();

		return $html;
	}

	/**
	 * Grid listing layout option 3.
	 *
	 * 
	 */
	public static function gridOption3(){
		$html = self::printCustomStyles() ;
		$html .= '<div class="fr-list-container pure_centered pure_row is-grid">'.self::liveSearch();
		$html .= '<div>';

		global $PcPlugin;
		$baseModule = $PcPlugin->getModule('base', 'BaseController');
		$basePublicController = $baseModule->getController('PublicController');

		$fundraisersModule = $PcPlugin->getModule('fundraisers', 'FundraisersController');
		$paginatorController = $fundraisersModule->getController('PaginatorController');

		$used = array();
		$counter = 1;
		foreach(self::$fundraisers->external_fundraisers as $fundraiser){
		if(!in_array($fundraiser->id, $used)){
			array_push($used, $fundraiser->id);

			$fundraiser->funding_goal = (
				$fundraiser->funding_goal !== 'anonymous' ?
				$fundraiser->funding_goal :
				$fundraiser->funding_total
			);
			
			$fundraiser->funding_needed = (
				$fundraiser->funding_needed !== 'anonymous' ?
				$fundraiser->funding_needed :
				$fundraiser->funding_total
			);

			$title = $fundraiser->name;
			if(isset(self::$options['title']) && self::$options['title'] == 'owner_name'){
			$title = $fundraiser->owner->name;
			}
			if(isset(self::$options['title']) && self::$options['title'] == 'title_and_owner_name'){
			$title = $fundraiser->name.'<br /> by '.$fundraiser->owner->name;
			}

			if ($fundraiser->images->medium == NULL) {
			$image = $fundraiser->images->large;
			}else{
			$image = $fundraiser->images->medium;
			}

			$funded = self::percent(($fundraiser->funding_goal-$fundraiser->funding_needed) ,$fundraiser->funding_goal);
			$html .= '
			<div class="fr-grid-list-item pure_span_8 pure_col no-border fundraiser_'.$fundraiser->id.'">
				<div class="fr-grid-list-content">
				<div class="fr-listing-avatar-container extended pure_span24">
					<div class="fr-listing-avatar" style="background-image: url('.$image.')">
						<a href="?fundraiser='.$fundraiser->slug.'&campaign='. self::$options['campaign'] .'" class="overlay-link pc-fundraiser-more-info"></a>
					</div>
				</div>
				<div class="fr-grid-item-content simplified pure_col pure_span_24">
					<div class="fr-grid-title-container">
					<p class="fr-grid-title extended simplified">'.$title.'</p>
					<p class="fr-grid-desc extended simplified">'.strip_tags(self::truncate($fundraiser->about, 150)).'</p>
					</div>
				</div>
				<div class="fr-actions extended simplified no-border pure_col pure_span_24">
					<a class="fr-themed-link pc-fundraiser-more-info" href="?fundraiser='.$fundraiser->slug.'&campaign='. self::$options['campaign'] .'">More</a>
					<a class="fr-themed-link" target="_blank" href="'.$basePublicController::pcUrl().'/fundraisers/'.$fundraiser->id.'/fund">Donate</a>
				</div>
				</div>
			</div>
			';
			if($counter %3 == 0){
			$html .= '<div class="clearfix"></div>';
			}
			$counter ++;
		}
		}

		$html .= self::listNotFound(!!(count($used)==0));
		$html .= '</div>';
		$html .= '</div>';
		$html .= $paginatorController::pageLinks(self::$fundraisers->meta);
		$html .= $basePublicController::poweredBy();

		return $html;
	}

	/**
	 * Grid listing layout option 4.
	 *
	 * 
	 */
	public static function gridOption4() {

		$html = self::printCustomStyles() ;
		$html .= '<div class="fr-list-container pure_centered pure_row is-grid">'.self::liveSearch().'</div>';
		$html .= '<div class="pure_col no-padding">';

		global $PcPlugin;
		$baseModule = $PcPlugin->getModule('base', 'BaseController');
		$basePublicController = $baseModule->getController('PublicController');

		$fundraisersModule = $PcPlugin->getModule('fundraisers', 'FundraisersController');
		$paginatorController = $fundraisersModule->getController('PaginatorController');

		$used = array();
		$counter = 1;
		foreach(self::$fundraisers->external_fundraisers as $fundraiser){
		if(!in_array($fundraiser->id, $used)){
			array_push($used, $fundraiser->id);

			$fundraiser->funding_goal = (
				$fundraiser->funding_goal !== 'anonymous' ?
				$fundraiser->funding_goal :
				$fundraiser->funding_total
			);
			
			$fundraiser->funding_needed = (
				$fundraiser->funding_needed !== 'anonymous' ?
				$fundraiser->funding_needed :
				$fundraiser->funding_total
			);

			self::$fundraiser = $fundraiser;

			$title = $fundraiser->name;
			if(isset(self::$options['title']) && self::$options['title'] == 'owner_name'){
			$title = $fundraiser->owner->name;
			}
			if(isset(self::$options['title']) && self::$options['title'] == 'title_and_owner_name'){
			$title = $fundraiser->name.'<br /> by '.$fundraiser->owner->name;
			}

			if ($fundraiser->images->large == NULL) {
			$image = $fundraiser->images->medium;
			}else{
			$image = $fundraiser->images->large;
			}

			$funded = self::percent(($fundraiser->funding_goal-$fundraiser->funding_needed) ,$fundraiser->funding_goal);
			$html .= '
			<div class="pure_span_8 pure_col no-border fundraiser_'.$fundraiser->id.'"">
				<div class="family">
				<a href="?fundraiser='. $fundraiser->slug .'&campaign='. self::$options['campaign'] .'" class="cover" style="background-image: url('. $image .');">
				</a>
				<div class="caption">
					<h3><a href="?fundraiser='. $fundraiser->slug .'&campaign='. self::$options['campaign'] .'">'. $title .'</a></h3>
					'. self::grid4Pieces('adopting_from') .'
					'. self::grid4Pieces('funding_list') .'
				</div>
				</div>
			</div>
			';

			if($counter %3 == 0){
			$html .= '<div class="clearfix"></div>';
			}
			$counter ++;
		}
		}
		$html .= '</div>';
		$html .= self::listNotFound(!!(count($used)==0));
		$html .= $paginatorController::pageLinks(self::$fundraisers->meta);
		$html .= $basePublicController::poweredBy();

		return $html;
	}

	/**
	 * Funding stats for grid listing.
	 *
	 * 
	 */
	public static function gridFundingStats($fundraiser, $layout = 1){
		$klass = ($fundraiser->funding_goal != 'anonymous' && ($fundraiser->recurring_funding_goal != NULL && $fundraiser->recurring_funding_goal != 'anonymous')) ? 'pure_span_12' : 'pure_span_24';
		$html = '';
		if($fundraiser->funding_goal != 'anonymous'){
		$funded = self::percent(($fundraiser->funding_goal-$fundraiser->funding_needed) ,$fundraiser->funding_goal);
		$html .= '
			<div class="pure_col '.$klass.'">
			<div class="fr-grid-status-title pure_col pure_span_24" title="'.$funded.'">
				<span>One-time donations:</span>
			</div>
			<div class="fr-grid-status pure_col pure_span_24" title="'.$funded.'">
				<div class="fr-grid-progress" style="width:'.$funded.'%"></div>
			</div>
			<div class="fr-grid-stats '.( $layout == 2 ? 'extended' : '' ).' pure_col pure_span_24">
				<p>
				$'.number_format(($fundraiser->funding_goal-$fundraiser->funding_needed), 0, '.', ',').'
				of
				$'.number_format($fundraiser->funding_goal, 0, '.', ',').'
				raised
				</p>
			</div>
			</div>
		';
		}

		if($fundraiser->recurring_funding_goal != NULL && $fundraiser->recurring_funding_goal != 'anonymous'){
		$funded = self::percent(($fundraiser->recurring_funding_goal-$fundraiser->recurring_funding_needed) ,$fundraiser->recurring_funding_goal);
		$html .= '
			<div class="pure_col '.$klass.'">
			<div class="fr-grid-status-title pure_col pure_span_24" title="'.$funded.'">
				<span>Recurring donations:</span>
			</div>
			<div class="fr-grid-status pure_col pure_span_24" title="'.$funded.'">
				<div class="fr-grid-progress" style="width:'.$funded.'%"></div>
			</div>
			<div class="fr-grid-stats '.( $layout == 2 ? 'extended' : '' ).' pure_col pure_span_24">
				<p>
				$'.number_format(($fundraiser->funding_goal-$fundraiser->funding_needed), 0, '.', ',').'
				of
				$'.number_format($fundraiser->funding_goal, 0, '.', ',').'
				raised
				</p>
			</div>
			</div>
		';
		}
		return $html;
	}

	/**
	 * List of Last Fundraisers.
	 *
	 * 
	 */
	public static function listingLastGrid() {

		$options = get_option( 'purecharity_plugin_fundraisers_settings' );

		global $PcPlugin;
		$baseModule = $PcPlugin->getModule('base', 'BaseController');
		$basePublicController = $baseModule->getController('PublicController');

		$html = self::printCustomStyles() ;
		$html .= '<div class="fr-list-container is-grid">';

		$used = array();
		foreach(self::$fundraisers->external_fundraisers as $fundraiser){
		if(!in_array($fundraiser->id, $used)){
			array_push($used, $fundraiser->id);

			$title = $fundraiser->name;
			if(isset(self::$options['title']) && self::$options['title'] == 'owner_name'){
			$title = $fundraiser->owner->name;
			}
			if(isset(self::$options['title']) && self::$options['title'] == 'title_and_owner_name'){
			$title = $fundraiser->name.'<br /> by '.$fundraiser->owner->name;
			}

			$html .= '
			<div class="fr-grid-list-item fundraiser_'.$fundraiser->id.'">
				<div class="fr-grid-list-content">
				<div class="fr-listing-avatar-container">
					<div class="fr-listing-avatar" style="background-image: url('.$fundraiser->images->large.')"></div>
					</div>
				<div class="fr-grid-item-content">
				<p class="fr-grid-title">'.$title.'</p>
				<p class="fr-grid-desc">'.$fundraiser->about.'</p>
				'.self::gridFundingStats($fundraiser).'
				</div>
				<div class="fr-actions pure_col pure_span_24">
				<a class="fr-themed-link pc-fundraiser-more-info" href="?fundraiser='.$fundraiser->slug.'&campaign='. self::$options['campaign'] .'">More</a>
				<a class="fr-themed-link" target="_blank" href="'.$basePublicController::pcUrl().'/fundraisers/'.$fundraiser->id.'/fund">Donate</a>
				</div>
			</div>
			</div>
			';
		}
		}

		$html .= self::listNotFound(false);
		$html .= '</div>';

		return $html;
	}

	/**
	 * Single fundraiser list item - layout 4
	 *
	 * 
	 */
	public static function featuredFundraiser(){

		$used = array();
		$counter = 1;

		self::$fundraiser->funding_goal = (
			self::$fundraiser->funding_goal !== 'anonymous' ?
			self::$fundraiser->funding_goal :
			self::$fundraiser->funding_total
		);
		
		self::$fundraiser->funding_needed = (
			self::$fundraiser->funding_needed !== 'anonymous' ?
			self::$fundraiser->funding_needed :
			self::$fundraiser->funding_total
		);

		$title = self::$fundraiser->name;
		if(isset(self::$options['title']) && self::$options['title'] == 'owner_name'){
		$title = self::$fundraiser->owner->name;
		}
		if(isset(self::$options['title']) && self::$options['title'] == 'title_and_owner_name'){
		$title = self::$fundraiser->name.'<br /> by '.self::$fundraiser->owner->name;
		}

		if (self::$fundraiser->images->large == NULL) {
		$image = self::$fundraiser->images->medium;
		}else{
		$image = self::$fundraiser->images->large;
		}

		$funded = self::percent((self::$fundraiser->funding_goal-self::$fundraiser->funding_needed) ,self::$fundraiser->funding_goal);
		$html = '
		<div class="pure_span_8 pure_col no-border fundraiser_'.self::$fundraiser->id.'"">
			<div class="family">
			<a href="'.self::$options['redirect'].'?fundraiser='. self::$fundraiser->slug .'" class="cover" style="background-image: url('. $image .');">
			</a>
			<div class="caption">
				<h3><a href="'.self::$options['redirect'].'?fundraiser='. self::$fundraiser->slug .'">'. $title .'</a></h3>
				'. self::grid4Pieces('adopting_from') .'
				<span class="raised">$'. number_format(self::$fundraiser->funding_goal-self::$fundraiser->funding_needed, 2).' Raised</span>
			</div>
			</div>
		</div>
		';

		return $html;
	}
}